class TextUtillity {
	constructor() {
		this.sentenceEndingPunctuation = ['. ', '! ', '? ', ":\n", ".\n", "!\n", "?\n"];
		this.sentenceEndingPunctuationSingle = ['.', '!', '?', ":"];
	}

	getSentences(text) {
		// Initialize an array to store the sentences.
		const sentences = [];

		// Initialize a variable to keep track of the start of the current sentence.
		let sentenceStart = 0;

		// Iterate through each character in the text.
		for (let i = 0; i < text.length; i++) {
			// Check if the current character is a sentence-ending punctuation mark.
			if (this.sentenceEndingPunctuation.includes(text[i] + text[i + 1])) {
				// Extract the sentence from the start to the current position.
				const sentence = text.slice(sentenceStart, i + 1).trim();
				// Check if the extracted sentence ends with a valid punctuation mark.
				if (this.sentenceEndingPunctuationSingle.includes(sentence.slice(-1))) {
					// Add the sentence to the array of valid sentences.
					sentences.push(sentence);
				}

				// Update the start of the next sentence.
				sentenceStart = i + 1;
			}
		}

		// Check if there is any remaining text after the last sentence.
		if (sentenceStart < text.length) {
			const lastSentence = text.slice(sentenceStart).trim();
			if (this.sentenceEndingPunctuationSingle.includes(lastSentence.slice(-1))) {
				sentences.push(lastSentence);
			}
		}

		return sentences;
	}


	getTextFromMarkdown(markdown) {
		return markdown.innerText;
	}
}

class Clipboard {
	constructor(){

	}

	copyToClipboard(text) {
		const textarea = document.createElement('textarea');
		textarea.value = text;
		document.body.appendChild(textarea);
		textarea.select();
		document.execCommand('copy');
		document.body.removeChild(textarea);
	}

}

class Notification {
	constructor() {
		this.audioPrinter = new Audio(chrome.runtime.getURL("sound/wait.mp3"));
		this.audioDone = new Audio(chrome.runtime.getURL("sound/done.mp3"));
	}

	playDone() {
		this.audioDone.src = chrome.runtime.getURL("sound/done.mp3");
		this.audioDone.play();
	}
	
	stopDone() {
		this.audioComm.src = "";
	}

	playPrinter() {
		this.audioPrinter = new Audio(chrome.runtime.getURL("sound/wait.mp3"));
		this.audioPrinter.loop = true;
		this.audioPrinter.play();
	}

	stopPrinter(){
		this.audioPrinter.src = "";
	}
}

class ConfigManager {
	constructor() {
		this.options = {};
		this.listeners = new Array();
	}

	addListener(listener) {
		this.listeners.push(listener);
	}

	getOptions() {
		return this.options;
	}

	notifyListeners() {
		for (var i = 0; i < this.listeners.length; i++) {
			try {
				this.listeners[i].update(this.options);
			} catch (e) {
				//console.log("Exception happened during config update");
				//console.log(e);
			}
		}
	}

	listen(callback) {
		var _this = this;
		chrome.storage.sync.onChanged.addListener(function (changes, namespace) {
			_this.updateConfig();
		});
		_this.updateConfig(callback);
	}

	updateConfig(callback) {
		var _this = this;
		chrome.storage.sync.get({
			promptTemplate: DEFAULT_PROMPT_TEMPLATE,
			voiceName: DEFAULT_VOICE_NAME,
			enableSpeak: DEFAULT_SPEAK_ENABLE,
			enableRewrite: DEFAULT_REWRITE_ENABLE,
			promptPage: DEFAULT_PROMPT_PAGE,
			token: '',
			clientId: ''
		}, function (items) {
			_this.options = items;
			_this.notifyListeners();
			if (callback != null) {
				callback();
			}
		});
	}
}

class TopButton {
	constructor(){
		this.options = {};
		this.running = false;
		this.ready = false;
		this.notification = null;
	}

	setNotification(notification) {
		this.notification = notification;
	}

	setReady(ready) {
		this.ready = ready;
	}

	update(options) {
		this.options = options;
		if (this.ready) {
			this.setup();
		}		
	}

	setRunning(running) {
		this.running = running;
		if (this.running) {
			jQuery('#thecircle').attr('fill', 'red');
		} else {
			jQuery('#thecircle').attr('fill', 'green');
		}
	}

	getPrompt() {
		return {
			prompt: this.options.promptPage,
			menuName: "Hit This Button"
		};
	}

	isVisible(el) {
		var visibility = false;
		try {
			if (
				window.getComputedStyle(el).visibility == "visible"
				&& window.getComputedStyle(el).display != "none"
				&& (el.offsetWidth > 0 || el.offsetHeight > 0)
				&& (el.getBoundingClientRect().width > 0 || el.getBoundingClientRect().height > 0)) {
				visibility = true;
			}
		} catch (e) {

		}		
		
		console.log(el);
		console.log(visibility);
		return visibility;
	}

	getTextBody(){
		var ps = document.getElementsByTagName("P");		
		var accum = "I can't find any readable paragraph in this page. Please try another page.";
		var plist = new Array();
		for (var i=0; i<ps.length; i++) {
			var curP = ps[i];
			console.log(curP.innerText);
			if (this.isVisible(curP)){
				var pCount2 = 0;
				for (var j=0; j<curP.parentNode.childNodes.length; j++) {
					var curPar = curP.parentNode.childNodes[j];
					console.log(curPar.nodeName);
					if (curPar.nodeName != "#text" && this.isVisible(curPar)) {
						if (curPar.nodeName.toLowerCase() == "p") {
							if (curPar.innerText.length > 40) {
								pCount2++;
							}					
						}
					}					
				}
				console.log(pCount2);

				if (pCount2 >= 1) {
					var foundParent = false;
					for (var j=0; j<plist.length; j++) {
						if (plist[0] === curP.parentNode) {
							foundParent = true;
							break;
						}
					}
					console.log("found parent " + foundParent);
					
					if (foundParent == false) {
						plist.push(curP.parentNode);
					}
				}				
			}
			
		}
		if (plist.length > 0) {
			accum = "";
			for (var i=0; i<plist.length; i++) {
				accum = accum + plist[i].innerText + "\n";
				plist[i].style.border = "4px solid yellow";
			}
		} else {
			var body = document.getElementsByTagName("body")[0];
			accum = body.innerText;
			body.style.border = "4px solid yellow";
		}		

		console.log(accum);

		return accum.substring(0,4000);
	}

	setup(valid){
		var _thisButton = this;
		if (document.location.href.indexOf("chat.openai.com") == -1) {
			var containerId = 'draggable-container';
			var container = document.getElementById(containerId);
			if (this.options.promptPage.length > 0) {	
				if (container != null) {
					//topButton.value = menuName;
					//topButton.name = menuName;
					container.title = this.getPrompt().menuName;
				} else {
					var fill = valid == null || valid == true ? "green" : "gray";
					var containerHTML = '<div title="'+this.getPrompt().menuName+'" id="draggable-container"><svg id="draggable-svg" width="50" height="50"><circle id="thecircle" cx="25" cy="25" r="20" fill="'+ fill + '" stroke="white" stroke-width="2" /></svg></div>';
					jQuery('body').append(containerHTML);
					jQuery('#' + containerId).click(function () {
						if (valid  == null || valid == true) {
							if (_thisButton.running == false) {
								console.log("set running true");							
								_thisButton.setRunning(true);
								_thisButton.notification.playPrinter();
								jQuery('#draggable-svg').attr('fill', 'red');
								var prompt = _thisButton.getPrompt().prompt;
								var body = _thisButton.getTextBody();
								prompt = prompt.replaceAll("{page}", "\n\n" + body);
								console.log("creating prompt and sending");
								chrome.runtime.sendMessage({ message: "PageAction", prompt: prompt }, function (response) {
									console.log("Background script responded:", response);
								});							
							} else {
								console.log("Please wait your previous request is still processing...");
							}
						} else {
							alert(DEFAULT_INVALID_TOKEN);
							window.location.href = chrome.runtime.getURL("options.html");
						}											
					});
				}
			} else {
				if (container != null) {
					jQuery(container).remove();
				}
			}
		}
	}
}

class EventNarrator {
	constructor(){
		this.index = 0;
		this.textUtility = new TextUtillity();
		this.options = {};
		this.speakManager = null;
		this.textBuffer = "";
		this.clipboard = new Clipboard();
	}

	setSpeakManager(speakManager) {
		this.speakManager = speakManager;
	}

	update(options) {
		this.options = options;
	}

	reset() {
		this.index = 0;
		this.textBuffer = "";
	}

	speak(text) {
		var _this = this;
		var sentences = this.textUtility.getSentences(text);
		//console.log(sentences);
		if (sentences.length > 0) {
			var textToSpeak = '';
			for (var i = this.index; i < sentences.length; i++) {
				textToSpeak = textToSpeak + ' ' + sentences[i];
				this.index = this.index + 1;
			}
			this.speakInternal(textToSpeak);			
		}
	}

	speakInternal(textToSpeak) {
		if (textToSpeak.length > 0) {
			this.textBuffer = this.textBuffer + textToSpeak;
			console.log(textToSpeak);
			this.speakManager.speak(textToSpeak, this.options.voiceName);
		}
	}

	copyText(){
		try {
			this.clipboard.copyToClipboard(this.textBuffer);
		} catch (e) {
			console.log(e);
		}
	}
}

class EventManager {
	constructor(){
		this.options = {};
		this.conversationId = null;
		this.eventNarrator = null;
		this.onStartCallback = null;
		this.topButton = null;
	}

	setTopButton(topButton) {
		this.topButton = topButton;
	}

	setEventNarrator(eventNarrator) {
		this.eventNarrator = eventNarrator;
	}

	onStart(onStartCallback) {
		this.onStartCallback = onStartCallback;
	}

	update(options) {
		this.options = options;
	}

	receive(event) {
		//console.log(event);
		if (this.conversationId == null) {
			if (event.complete == false) {
				this.conversationId = event.data.conversation_id;
				this.eventNarrator.reset();
				this.onStartCallback();
			}
		}

		if (event.complete == false) {
			if (this.conversationId == event.data.conversation_id) {
				if (event.data.message != null && event.data.message.author.role == "assistant") {
					this.eventNarrator.speak(event.data.message.content.parts[0]);
				} else if (event.data.error != null) {
					this.eventNarrator.speak(event.data.error);
				}
			} else {
				//console.log("event is rejected different conversation id");
				//console.log(event);
			}
		} else {
			this.conversationId = null;
			this.eventNarrator.copyText();
			this.eventNarrator.reset();
			this.topButton.setRunning(false);			
		}
	}
}

class CSS {
	constructor(){

	}

	setup() {
		// content.js

		// Create a <style> element
		const style = document.createElement('style');
		style.textContent = `
		/* Your CSS styles go here */
		
		
		#draggable-container {
			position: fixed;
			bottom: 0;
			right: 0;
			margin: 20px;
			z-index: 99999999;
		}
		`;

		// Append the <style> element to the document's <head>
		document.head.appendChild(style);
	}
}

class ContentScript {
	constructor() {
		this.configManager = new ConfigManager();
		this.topButton = new TopButton();		

		this.notification = new Notification();
		this.eventManager = new EventManager();
		this.eventNarrator = new EventNarrator();
		this.speakManager = new SpeakManager({
			callback: function () {
				_this.notification.playDone();
			}
		});
		this.topButton.setNotification(this.notification);
		this.eventManager.setTopButton(this.topButton);
		this.eventNarrator.setSpeakManager(this.speakManager);
		this.eventManager.setEventNarrator(this.eventNarrator);

		this.configManager.addListener(this.topButton);
		this.configManager.addListener(this.eventManager);
		this.configManager.addListener(this.eventNarrator);
		var _this = this;
		
		this.speakManager.setPlayCallback(function () {
			_this.notification.stopPrinter();
		});

		this.eventManager.onStart(function () {
			_this.speakManager.clear();
			_this.speakManager.stop();
			//_this.notification.playPrinter();
		});

		this.css = new CSS();
	}

	listen(){
		var _this = this;
		chrome.runtime.onMessage.addListener(function (message, sendercrx, sendResponse) {
			if (message.command == "GetBody") {
				console.log("GetBodyRequest");
				var body = _this.topButton.getTextBody();
				sendResponse({ body: body});
			} else if (message.command == "ReceiveEvent") {
				console.log("receive event");
				_this.eventManager.receive(message.event);
			} else if (message.command == "ShowMessage") {
				console.log("show message");
				alert(message.prompt);
				_this.topButton.setRunning(false);
				_this.notification.stopPrinter();
			} else if (message.command == "ValidationResult") {
				console.log("ValidationResult", message.valid);
				_this.topButton.setReady(message.valid);
				_this.topButton.setup(message.valid);				
			}
		});
	}

	disableCsp(){
		chrome.runtime.sendMessage({ message: "DisableCsp" }, function (response) {
			console.log("Background script responded:", response);
		});
	}

	validateToken(callback) {
		chrome.runtime.sendMessage({ message: "ValidateToken" }, function (response) {
			console.log("Background script responded:", response);
			callback(response);
		});
	}

	init(){
		console.log("init");
		var _this = this;
		console.log("before disable csp");
		this.disableCsp();
		console.log("before config setup");
		this.configManager.listen(function (){
			console.log("config loaded");
			_this.css.setup();
			console.log("config setup button");
			console.log("script listen");
			_this.listen();
			_this.validateToken(function (response) {
				console.log(response);
			});			
		});		
	}
}

function ready(callback){
    // in case the document is already rendered
    if (document.readyState!='loading') callback();
    // modern browsers
    else if (document.addEventListener) document.addEventListener('DOMContentLoaded', callback);
    // IE <= 8
    else document.attachEvent('onreadystatechange', function(){
        if (document.readyState=='complete') callback();
    });
}

ready(function(){
	setTimeout(function () {
		const contentScript = new ContentScript();
		contentScript.init();
	}, 1000);		
});